#region References

using System;
using System.Collections;
using System.Configuration;
using System.Text;
using gov.va.med.vbecs.Common.Log;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{ 
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>David Askew</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/23/2008</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>This class handles incoming Blood Product Transfusion/Disposition HL7 messages.</summary>

	#endregion

	/// <summary>
	/// New class for BCE (CR 2962)
	/// CR 2985: Updated to provide better error handling of missing\invalid message data
	/// </summary>
	public sealed class BceHL7Parser
	{
		#region Inner Classes

		/// <summary>
		/// Used to hold data parsed from BTS O31 message
		/// </summary>
		public sealed class PatientTransfusionData
		{
			#region Variables

			private char _reactionMarker;
			
			private int _transmitCount;

			private string _bedsideVerificationUnitManualEntryReason;
			private string _commentSeparator;
			private string _divisionCode;
			private string _errorMessage;
			private string _locationName;
			private string _patientTransfusionComment;
			private string _primaryTransfusionist;
			
			private Guid _bloodUnitGuid;
			private Guid _patientGuid;

			private DAL.HL7.OpenLibrary.AckCodes _ackCode;

			#endregion

			#region Properties

			/// <summary>
			/// 
			/// </summary>
			internal DAL.HL7.OpenLibrary.AckCodes AckCode
			{
				get
				{
					return this._ackCode;
				}
				set
				{
					this._ackCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string BedsideVerificationUnitManualEntryReason
			{
				get
				{
					return this._bedsideVerificationUnitManualEntryReason;
				}
				set
				{
					this._bedsideVerificationUnitManualEntryReason = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal Guid BloodUnitGuid
			{
				get
				{
					return this._bloodUnitGuid;
				}
				set
				{
					this._bloodUnitGuid = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string DivisionCode
			{
				get
				{
					return this._divisionCode;
				}
				set
				{
					this._divisionCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string CommentSeparator
			{
				get
				{
					return this._commentSeparator;
				}
				set
				{
					this._commentSeparator = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string ErrorMessage
			{
				get
				{
					return this._errorMessage;
				}
				set
				{
					if ( this._errorMessage != null && this._errorMessage.Length > 0 )
					{
						this._errorMessage = string.Concat( this._errorMessage, "\n", value );
					}
					else
					{
						this._errorMessage = value;
					}
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string LocationName
			{
				get
				{
					return this._locationName;
				}
				set
				{
					this._locationName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal Guid PatientGuid
			{
				get
				{
					return this._patientGuid;
				}
				set
				{
					this._patientGuid = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientTransfusionComment
			{
				get
				{
					return this._patientTransfusionComment;
				}
				set
				{
					this._patientTransfusionComment = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PrimaryTransfusionist
			{
				get
				{
					return this._primaryTransfusionist;
				}
				set
				{
					this._primaryTransfusionist = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal char ReactionMarker
			{
				get
				{
					return this._reactionMarker;
				}
				set
				{
					this._reactionMarker = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal int TransmitCount
			{
				get
				{
					return this._transmitCount;
				}
				set
				{
					this._transmitCount = value;
				}
			}

			#endregion
		}

		/// <summary>
		/// Used to generate BRT~O32 response to BTS~O31 HL7 messages
		/// CR 2962
		/// </summary>
		private sealed class BceBrtMessage
		{
			#region Variables

			private const string ACK_MESSAGE_TYPE = "BRT~O32";
			private const string MESSAGE_TYPE = "BTS~O31";
			private const string CARRIAGE_RETURN = "\x0D";

			#endregion
		
			#region HL7 Acknowledgement Methods

			/// <summary>
			/// Builds an Application ackowledgement message
			/// </summary>
			internal static HL7BrtMessage BuildApplicationAcknowledgement(HL7BtsMessage originalMessage, AckCodes ackCode, string errSegments)
			{
				if( originalMessage == null )
				{
					throw( new ArgumentNullException( "originalMessage" ) );
				}
				//
				HL7Interface intParms = new HL7Interface( InterfaceName.BCE_COTS.ToString() );
				//
				StringBuilder ack = new StringBuilder();
				//
				ack.Append(BuildMSHSegment(intParms));
				ack.Append(BuildMSASegment(intParms, originalMessage, ackCode));
				//
				if ( errSegments != null && errSegments.Length > 0  )
				{
					ack.Append( errSegments );
				}
				//
				HL7ProtocolMessage ackMessage = VbecsHL7Bridge.BuildHL7ProtocolMessage( ack.ToString() );
				//
				return (HL7BrtMessage)ackMessage;
			}

			/// <summary>
			///	Build the MSH message segment
			/// </summary>
			private static string BuildMSHSegment(HL7Interface intParms)
			{
                //Defect 228028 - getting the current date/time for the message
                string msgDateTime = System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":", "");

				StringBuilder msh = new StringBuilder();
				//
				msh.Append("MSH");																	// MSH Segment ID
				msh.Append(intParms.FieldSeparator);												// Field Separator
				msh.Append(intParms.EncodingCharacters);											// Encoding Characters
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.VbecsApplicationId.Trim());										// Sending Application
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.VbecsFacilityId.Trim());										// Sending Facility
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.InterfaceApplicationId.Trim());									// Receiving Application
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.InterfaceFacilityId.Trim());									// Receiving Facility
				msh.Append(intParms.FieldSeparator);
                //Defect 228028
                msh.Append(msgDateTime);															// HL7 DateTime
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.FieldSeparator);
				msh.Append(ACK_MESSAGE_TYPE);														// HL7 Message Type
				msh.Append(intParms.FieldSeparator);
				msh.Append("VBECS").Append(System.DateTime.Now.ToString("yMMddhhssffff"));			// Message Control ID
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.ProcessingId);													// Processing ID
				msh.Append(intParms.FieldSeparator);
				msh.Append(intParms.VersionId.Trim());												// HL7 Version ID
				msh.Append(CARRIAGE_RETURN);
				//
				return msh.ToString();
			}

			/// <summary>
			/// Build the MSA message segment
			/// </summary>
			private static string BuildMSASegment(HL7Interface intParms, HL7BtsMessage originalMessage, AckCodes ackCode)
			{
				StringBuilder msa = new StringBuilder();
				//
				msa.Append("MSA");
				msa.Append(intParms.FieldSeparator);
				msa.Append(ackCode);
				msa.Append(intParms.FieldSeparator);
				msa.Append(originalMessage.GetMessageControlID());
				msa.Append(CARRIAGE_RETURN);
				//
				return msa.ToString();
			}

			/// <summary>
			/// Builds HL7 version 2.5 compliant ERR Segment
			/// </summary>
			internal static string BuildErrSegment( 
				string fieldSeparator,
				char componentSeparator,
				string segmentID, 
				int segmentSequence,  
				int fieldPosition, 
				int componentNumber, 
				int subComponentNumber, 
				AckErrorCodes ackErrorCode, 
				AckErrorSeverity ackErrorSeverity, 
				string errorMessage )
			{
				StringBuilder errSegment = new StringBuilder();
				//
				errSegment.Append( "ERR" );
				errSegment.Append( fieldSeparator );		// ERR.1
				errSegment.Append( fieldSeparator );		// ERR.2
				// ******************************************************************************************************************
				// ERR.2.1: segment ID (Specifies the 3-letter name for the segment.)
				if ( segmentID != null )
				{
					errSegment.Append( segmentID );	
				}
				errSegment.Append( componentSeparator );	
				// ******************************************************************************************************************
				// ERR.2.2: segment sequence (Identifies the segment occurrence within the message.)
				errSegment.Append( segmentSequence );		
				errSegment.Append( componentSeparator );	
				// ******************************************************************************************************************
				// ERR.2.3: field position (Identifies the number of the field within the segment. The first field 
				// is assigned a number of 1.)
				if ( fieldPosition > 0 )
				{
					errSegment.Append( fieldPosition );
				}
				errSegment.Append( componentSeparator );	// ERR.2.4
				errSegment.Append( componentSeparator );	
				// ******************************************************************************************************************
				// ERR.2.5: component number (Identifies the number of the component within the field. The first component 
				// is assigned a number of 1. Component number should not be specified when referring to the entire field.)
				if ( componentNumber > 0 )
				{
					errSegment.Append( componentNumber );
				}
				errSegment.Append( componentSeparator );	// ERR.2.6
				// ******************************************************************************************************************
				// ERR.2.6: sub-component number (Identifies the number of the sub-component within the component. The first 
				// sub-component is assigned a number of 1.  Sub-component number should not be specified when referring to the entire 
				// component.)
				if ( subComponentNumber > 0 )
				{
					errSegment.Append( subComponentNumber );
				}
				errSegment.Append( fieldSeparator );		// ERR.3
				// ******************************************************************************************************************
				// ERR.3: HL7 Error Code (Identifies the HL7 (communications) error code. Refer to HL7 Table 0357 _ Message Error 
				// Condition Codes for valid values.)
				// ERR.3.1: identifier
				errSegment.Append( HL7Utility.GetAckErrorCodeFromEnum(ackErrorCode) );	// ERR.3.1
				//
				errSegment.Append( componentSeparator );	// ERR.3.2
				// ******************************************************************************************************************
				// ERR.3.2: text
				if ( errorMessage != null && errorMessage != string.Empty )
				{
					errSegment.Append( errorMessage.Length > MAX_ERROR_MESSAGE_SIZE ? errorMessage.Substring(0, MAX_ERROR_MESSAGE_SIZE) : errorMessage );
				}
				//
				errSegment.Append( fieldSeparator );		// ERR.4
				// ******************************************************************************************************************
				// ERR.4: Severity (Identifies the severity of an application error. Knowing if something is Error, Warning or Information is intrinsic 
				// to how an application handles the content. Refer to HL7 Table 0516 - Error severity for valid values. If ERR-3 has a value of "0", ERR-4 
				// will have a value of "I".)
				errSegment.Append( HL7Utility.GetAckErrorSeverityFromEnum(ackErrorSeverity) );	// ERR.4
				//
				errSegment.Append( CARRIAGE_RETURN );
				//
				return errSegment.ToString();
			}

			#endregion
		}

		#endregion

		#region Variables

		private const string COMMENT_SEPARATOR_CONFIG_KEY = "BceTransfusionCommentSeparator";
		private const string REACTION_AFFIRMATIVE_INDICATOR = "Yes";
		private const string REACTION_MARKER_CONFIG_KEY = "BceTransfusionReactionMarker";
		private const string RECORD_LOCKED = "Unable to process update: application record is locked.";
		private const string RECORD_LOCKED_MAX_ATTEMPTS_MADE = "Unable to process update: The application record is locked and the maximum number of retransmit attempts has been made.";
		private const string REGEXPATTERN = "(?<=(<{2}|^)).+?(?=(>{2}|$))";
		private const string VBECS_USER = "VBECS";

		private const int MAX_ERROR_MESSAGE_SIZE = 300;

		private const string ACK_MESSAGE_TYPE = "ACK";
		private const string CARRIAGE_RETURN = "\x0D";
		private const string EMPTY_LOCATION_1 = "''";
		private const string EMPTY_LOCATION_2 = "\"\"";

	    private const int LOCK_FORM_ID = 314;   //Locking
		/// <summary>
		/// Generic error message template
		/// </summary>
		private const string ERROR_MESSAGE_BASE = "Error processing HL7 message: \n\n";
		/// <summary>
		/// HL7 error message template
		/// </summary>
		private const string ERROR_MESSAGE_HL7 = "Missing or invalid content in HL7 message: \n\n";
		/// <summary>
		/// Generic no-details error message
		/// </summary>
		private const string ERROR_MESSAGE_NO_DETAILS = "No details available.";

		/// <summary>
		/// CR 3006
		/// </summary>
		private const string ID_MSH = "MSH";
		private const string ID_PID = "PID";
		private const string ID_PV1 = "PV1";
		private const string ID_ORC = "ORC";
		private const string ID_BPO = "BPO";
		private const string ID_BTX = "BTX";
		private const string ID_NTE_RE = "NTE_RE";
		private const string ID_NTE_VA_BCR = "NTE_VA_BCR";

		/// <summary>
		/// CR 3006
		/// </summary>
		private const int SEQUENCE_MSH = 1;
		private const int SEQUENCE_PID = 2;
		private const int SEQUENCE_PV1 = 3;
		private const int SEQUENCE_ORC = 4;
		private const int SEQUENCE_BPO = 5;
		private const int SEQUENCE_BTX = 6;
		private const int SEQUENCE_NTE_RE = 7;
		private const int SEQUENCE_NTE_VA_BCR = 8;

		/// <summary>
		/// CR 3006
		/// </summary>
		private const char SEPARATOR = ':';

        // Events Logger
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8740"> 
		///		<ExpectedInput>Valid HL7Interface, Valid HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AA" Ack code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8741"> 
		///		<ExpectedInput>Null HL7Interface, Valid HL7 message string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8777"> 
		///		<ExpectedInput>Valid HL7Interface, Null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8778"> 
		///		<ExpectedInput>Null HL7Interface, Null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		/// 
		///<Case type="1" testid ="8856"> 
		///		<ExpectedInput>Invalid field separator (MSH[1])</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8857"> 
		///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[0]</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8858"> 
		///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[1]</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8915"> 
		///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[2]</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8919"> 
		///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[3]</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8920"> 
		///		<ExpectedInput>Valid HL7Interface, Invalid ADT Message (missing and invalid segment data)</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AR" Ack code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8921"> 
		///		<ExpectedInput>Valid HL7Interface, Valid HL7 message string with locking condition in place</ExpectedInput>
		///		<ExpectedOutput>Valid HL7ProtocolMessage with "AE" Ack code</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Primary method to handle inbound BTS O31 patient transfusion\disposition HL7 messages from the BCE Interface 
		/// and insert the new\updated patient transfusion record in the VBECS database.
		/// </summary>
		/// <param name="hl7Interface"><see cref="HL7Interface"/>containing interface parameters for the BCE COTS interface</param>
		/// <param name="message">HL7 message string</param>
		/// <returns>HL7ProtocolMessage</returns>
		public static HL7ProtocolMessage ParseHL7Message( HL7Interface hl7Interface, string message )
		{
			#region Handle Invalid Parameters

			if( hl7Interface == null )
			{
				throw( new ArgumentNullException("intParms") );
			}
			//
			if( message == null )
			{
				throw( new ArgumentNullException("message") );
			}

			#endregion
			//
			#region Variables

			string errData = null;
			string errMessage = null;
			//
			HL7ProtocolMessage ackMessage = null;
			//
			HL7BtsMessage hl7BtsMessage = null;
			//
			PatientTransfusionData patientTransfusionData = null;
			//
			Common.MessageStatus messageStatus = MessageStatus.ProcessingIncomingMessage;
			//
			DAL.HL7.OpenLibrary.AckCodes ackCode = AckCodes.AA;

			#endregion
			//
			try
			{
				hl7BtsMessage = new HL7BtsMessage( message );
				//
				if (hl7BtsMessage.AckErrorCode == AckErrorCodes.MessageAccepted)
				{
					patientTransfusionData = new PatientTransfusionData();
					//
                    patientTransfusionData.CommentSeparator = Common.VbecsConfig.GetConfigValueFromDB(COMMENT_SEPARATOR_CONFIG_KEY, Common.ApplicationNameGlobalConfig.HL7Service);
                    patientTransfusionData.ReactionMarker = Convert.ToChar(Common.VbecsConfig.GetConfigValueFromDB(REACTION_MARKER_CONFIG_KEY, Common.ApplicationNameGlobalConfig.HL7Service));
					//
					errData = ValidateMessageContent( hl7Interface, hl7BtsMessage, ref patientTransfusionData );
					//
					// Check message to ensure segment data has required fields
					if ( errData == string.Empty )
					{
						HL7MessageLog.InsertMessageLog( HL7MessageLog.GetMessageDataForMessageLogInsert( message, messageStatus, hl7Interface, Common.UpdateFunction.HL7PatientUpdateInterface ) );
						//
						errData = ProcessUpdatedData( hl7Interface, hl7BtsMessage, ref patientTransfusionData );
						//
						if ( errData == string.Empty )
						{
							messageStatus = MessageStatus.SuccessfullyCompleted;
						}
						else
						{
							// Locking condition or broken rule
							ackCode = patientTransfusionData.AckCode;
							errMessage = string.Concat( ERROR_MESSAGE_BASE, patientTransfusionData != null ? patientTransfusionData.ErrorMessage : ERROR_MESSAGE_NO_DETAILS );
							messageStatus = MessageStatus.ProcessingError;
						}
					}
					else
					{
						ackCode = AckCodes.AR;
						errMessage = string.Concat( ERROR_MESSAGE_HL7, errData );
						messageStatus = MessageStatus.ProcessingError;
					}
				}
				else
				{
					ackCode = AckCodes.AR;
					errData = BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, 0, 0, hl7BtsMessage.AckErrorCode, AckErrorSeverity.Error, string.Empty);
					errMessage = hl7BtsMessage.AckErrorCode.ToString();
					messageStatus = MessageStatus.ProcessingError;
				}
			}
			catch(Exception ex)
			{
				ackCode = AckCodes.AR;
				errData = BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, 0, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, null);
				errMessage = string.Concat( ERROR_MESSAGE_BASE, HL7Utility.GetInnerExceptionFromException(ex) );
				messageStatus = MessageStatus.ProcessingError;
			}
			//
			#region Handle Acknowedgement

			ackMessage = BceBrtMessage.BuildApplicationAcknowledgement( hl7BtsMessage, ackCode, errData ); // CR 2985 
			//CR 3554
		    if (hl7BtsMessage != null)
		    {
		        HL7MessageLog.InsertAckMessageControlId( hl7BtsMessage.MessageControlID, ackMessage.GetMessageControlID(), messageStatus, hl7Interface.InterfaceName, Common.UpdateFunction.HL7BceInterface );
		        //
		        if ( ackCode == AckCodes.AE || ackCode == AckCodes.AR )
		        {
		            // CR 3026: If AR message we don't request message be re-sent 
                    // CR 3554
		            if ( ackCode == AckCodes.AR ) {
		                if (patientTransfusionData != null) patientTransfusionData.TransmitCount = 0;
		            } 
		            //
		            // CR 2972: AE => Locking error, update transmit count and try again; AR => just update message status
		            // CR 3026: If AE or AR, need to update message status
                    // CR 3554
		            if (patientTransfusionData != null)
		                HL7MessageLog.UpdateMessageStatus( messageStatus, hl7BtsMessage.GetMessageControlID(), errMessage, patientTransfusionData.TransmitCount, hl7Interface.InterfaceName, Common.UpdateFunction.HL7BceInterface );
		            //
		            // CR 3026: If AE or AR, need to write error to event log
		            // EventLogAppender
		            _eventsLogger.Error("VBECS BCE HL7 Parser: " + errMessage);
		            //System.Diagnostics.EventLog.WriteEntry( "VBECS BCE HL7 Parser", errMessage, System.Diagnostics.EventLogEntryType.Error );

		            //
		            try
		            {
		                // CR 3026: If AE or AR, need to send email to interface admin
		                new HL7MailMessage().SendMessage(hl7Interface.InterfaceAdministratorEmail, PII                        ", "BCE HL7 Parser Error", errMessage, "smtp.DNS   ");
		            }
		            catch( Exception mailexc )
		            {
		                var mailerrmsg = "Exception encountered sending mail message to Interface Administrator.   Details:\n\n\n" + HL7Utility.GetInnerExceptionFromException(mailexc);
		                // EventLogAppender
		                _eventsLogger.Error("VBECS BCE HL7 Parser: " + mailerrmsg);
		            }
		        }
		    }
		    //
			return ackMessage;

			#endregion
		}

		#endregion

		#region Private Methods
		
		#region Validate Message

		/// <summary>
		/// CR 3006
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <param name="patientTransfusionData"></param>
		/// <returns></returns>
		public static string ValidateMessageContent( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage, ref PatientTransfusionData patientTransfusionData )
		{
			StringBuilder errSegments = new StringBuilder();
			//
			try
			{
				// Validate the required message segments that VBECS uses.
				//
				errSegments.Append( ValidateMshSegmentContent(hl7Interface, hl7BtsMessage) );
				errSegments.Append( ValidatePidSegmentContent(hl7Interface, hl7BtsMessage) );
				errSegments.Append( ValidatePv1SegmentContent(hl7Interface, hl7BtsMessage) );
				errSegments.Append( ValidateBtxSegmentContent(hl7Interface, hl7BtsMessage) );
				//
				if ( errSegments.Length == 0 )
				{
					// Now attempt to load a PatientTransfusionData object with the message data.
					// The Bts message structure has passed validation and now we need to check the content.
					//
					errSegments.Append( ValidatePatientTransfusionData(hl7Interface, hl7BtsMessage, ref patientTransfusionData) ); 
				}
			}
			catch(HL7Exception hl7Ex)
			{
				// This means critical data is missing from message and we cannot continue.
				//
				string segmentId = null;
				int segmentSequence = 0;
				int fieldPosition = 0;
				//
				if ( hl7Ex.Id != null )
				{
					string [] segmentErrorData = hl7Ex.Id.ToString().Split( new char[]{ SEPARATOR });
					//
					segmentId = segmentErrorData[0];
					segmentSequence = segmentErrorData.Length > 1 ? Convert.ToInt32(segmentErrorData[1]) : 0;
					fieldPosition = segmentErrorData.Length > 2 ? Convert.ToInt32(segmentErrorData[2]) : 0;
				}
				//
				errSegments.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], segmentId, segmentSequence, fieldPosition, 0, 0, 
					AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(hl7Ex)) );
			}
			//
			return errSegments.ToString();
		}	

		/// <summary>
		/// CR 3006: added try/catch blocks
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <returns></returns>
		private static string ValidateMshSegmentContent( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage )
		{
			StringBuilder errSegment = new StringBuilder();
			//
			// Make sure MSH is present ... 
			if ( hl7BtsMessage.MSH == null || hl7BtsMessage.MSH.Length == 0 )
			{
				throw new HL7Exception("MSH Segment data not found; processing cannot continue.", string.Concat(ID_MSH, SEPARATOR, SEQUENCE_MSH));
			}
			else
			{
				// *********************************************************************************************************************************************************************
				// Field Separator
				if ( hl7BtsMessage.MSH[1] == null || hl7BtsMessage.MSH[1].Length < 1 )
				{
					throw new HL7Exception("Field Separator is missing; processing cannot continue.", string.Concat(ID_MSH, SEPARATOR, SEQUENCE_MSH, SEPARATOR, 1));
				}
				else if ( hl7BtsMessage.MSH[1] != hl7Interface.FieldSeparator )
				{
					throw new HL7Exception("Field Separator is invalid; processing cannot continue.", string.Concat(ID_MSH, SEPARATOR, SEQUENCE_MSH, SEPARATOR, 1));
				}
				// *********************************************************************************************************************************************************************
				// Encoding Characters
				if ( hl7BtsMessage.MSH.Length < 2 || hl7BtsMessage.MSH[2] == null || hl7BtsMessage.MSH[2].Length < 4 )
				{
					throw new HL7Exception("Encoding Characters are missing; processing cannot continue.", string.Concat(ID_MSH, SEPARATOR, SEQUENCE_MSH, SEPARATOR, 2));
				}
				else
				{
					bool encodingCharInvalid = false;
					char [] encodingCharacters = hl7BtsMessage.MSH[2].Length == 4 ? hl7BtsMessage.MSH[2].ToCharArray(0, 4) : null;
					if ( encodingCharacters != null )
					{
						for ( int idx = 0; idx < encodingCharacters.Length && idx < encodingCharacters.Length; idx++ )
						{
							if ( encodingCharacters[idx] != hl7Interface.EncodingCharacters[idx] )
							{
								encodingCharInvalid = true;
								break;
							}
						}
					}
					else
					{
						encodingCharInvalid = true;
					}
					//
					if ( encodingCharInvalid )
					{
						throw new HL7Exception("One or more Encoding Characters are invalid; processing cannot continue.", string.Concat(ID_MSH, SEPARATOR, SEQUENCE_MSH, SEPARATOR, 2));
					}
				}
				// *********************************************************************************************************************************************************************
				// Sending Application
				if ( hl7BtsMessage.MSH.Length < 4 || hl7BtsMessage.MSH[3] == null || hl7BtsMessage.MSH[3].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						3, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// Receiving Application
				if ( hl7BtsMessage.MSH.Length < 6 || hl7BtsMessage.MSH[5] == null || hl7BtsMessage.MSH[5].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						5, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// Date/Time Of Message
				if ( hl7BtsMessage.MSH.Length < 8 || hl7BtsMessage.MSH[7] == null || hl7BtsMessage.MSH[7].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						7, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// Message Type
				if ( hl7BtsMessage.MSH.Length < 10 || hl7BtsMessage.MSH[9] == null || hl7BtsMessage.MSH[9].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						9, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					try
					{
						if ( hl7BtsMessage.MessageType == null )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
								9, 1, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
						}
						else if ( !hl7BtsMessage.IsMessageTypeValid() )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
								9, 1, 0, AckErrorCodes.UnsupportedMessageType, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch (Exception ex)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
							9, 1, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
					}
					//
					try
					{
						if ( hl7BtsMessage.TriggerEvent == null )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
								9, 2, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
						}
						else if ( !hl7BtsMessage.IsTriggerEventValid() )
						{ 
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
								9, 2, 0, AckErrorCodes.UnsupportedMessageType, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch (Exception ex)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
							9, 2, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
					}
				}
				// *********************************************************************************************************************************************************************
				// Message Control ID
				if ( hl7BtsMessage.MSH.Length < 11 || hl7BtsMessage.MSH[10] == null || hl7BtsMessage.MSH[10].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						10, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// Processing ID
               if ( hl7BtsMessage.MSH.Length < 12 || hl7BtsMessage.MSH[11] == null || hl7BtsMessage.MSH[11].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						11, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
                //CR 3243
                else
                {
                    char[] procId = hl7BtsMessage.MSH[11].ToCharArray();
                    if (procId[0] != hl7Interface.ProcessingId)
                    {
                        errSegment.Append(BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 11, 0, 0, AckErrorCodes.ConflictingProcessingId, AckErrorSeverity.Error, string.Empty));
                    }
                }       
              
				// *********************************************************************************************************************************************************************
				// Version ID
				if ( hl7BtsMessage.MSH.Length < 13 || hl7BtsMessage.MSH[12] == null || hl7BtsMessage.MSH[12].Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						12, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else if ( hl7BtsMessage.MSH[12] != hl7Interface.VersionId )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_MSH, SEQUENCE_MSH, 
						12, 1, 0, AckErrorCodes.UnsupportedVersionId, AckErrorSeverity.Error, string.Empty) );
				}
			}
			//
			return errSegment.ToString();
		}

		/// <summary>
		/// CR 3006: added try/catch blocks
		/// CR 3186: removed validation of non-required fields: patient gender and dob.
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <returns></returns>
		private static string ValidatePidSegmentContent( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage )
		{
			StringBuilder errSegment = new StringBuilder();
			//
			// Make sure PID is present ... 
			if ( hl7BtsMessage.PID == null || hl7BtsMessage.PID.Length == 0 )
			{
				throw new HL7Exception("PID Segment data not found; processing cannot continue.", string.Concat(ID_PID, SEPARATOR, SEQUENCE_PID));
			}
			else
			{
				// Patient DFN (VistAPatientId)
				if ( hl7BtsMessage.PID.Length < 4 || hl7BtsMessage.PID[3] == null || hl7BtsMessage.PID[3] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
						3, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// Patient Name
				if ( hl7BtsMessage.PID.Length < 6 || hl7BtsMessage.PID[5] == null || hl7BtsMessage.PID[5] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
						5, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					try
					{
						string [] patientName = HL7Utility.ParseGetStringArray( hl7BtsMessage.PID[5], hl7BtsMessage.Delimiters[1].ToString() );
						//
						string patientLastName = HL7Utility.ConvertString( patientName[0] );
						string patientFirstName = HL7Utility.ConvertString( patientName[1] );
						//
						if ( patientLastName == null )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
								5, 1, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
						}
						//
						if ( patientFirstName == null )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
								5, 2, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch (Exception ex)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
							5, 0, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
					}
				}
				// *********************************************************************************************************************************************************************
				// Patient SSN
				if ( hl7BtsMessage.PID.Length < 20 || hl7BtsMessage.PID[19] == null || hl7BtsMessage.PID[19] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
						19, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
			}
			//
			return errSegment.ToString();
		}

		/// <summary>
		/// CR 3006: added try/catch blocks
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <returns></returns>
		private static string ValidatePv1SegmentContent( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage )
		{
			StringBuilder errSegment = new StringBuilder();
			//
			// Make sure PV1 is present ... 
			if ( hl7BtsMessage.PV1 == null || hl7BtsMessage.PV1.Length == 0 )
			{
				throw new HL7Exception("PV1 Segment data not found; processing cannot continue.", string.Concat(ID_PV1, SEPARATOR, SEQUENCE_PV1));
			}
			else
			{
				if ( hl7BtsMessage.PV1.Length < 4 || hl7BtsMessage.PV1[3] == null || hl7BtsMessage.PV1[3] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PV1, SEQUENCE_PV1,
						3, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
			}
			//
			return errSegment.ToString();
		}

		/// <summary>
		///  CR 3006: added try/catch blocks
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <returns></returns>
		private static string ValidateBtxSegmentContent( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage )
		{
			StringBuilder errSegment = new StringBuilder();
			//
			// Make sure BTX is present ... 
			if ( hl7BtsMessage.BTX == null || hl7BtsMessage.BTX.Length == 0 )
			{
				throw new HL7Exception("BTX Segment data not found; processing cannot continue.", string.Concat(ID_BTX, SEPARATOR, SEQUENCE_BTX));
			}
			else
			{
				// BC Donation ID (BloodUnitGuid)
				if ( hl7BtsMessage.BTX.Length < 3 || hl7BtsMessage.BTX[2] == null || hl7BtsMessage.BTX[2] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						2, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// BP Amount
				if ( hl7BtsMessage.BTX.Length < 10 || hl7BtsMessage.BTX[9] == null || hl7BtsMessage.BTX[9] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						9, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// BP Transfusion/Disposition Status
				if ( hl7BtsMessage.BTX.Length < 12 || hl7BtsMessage.BTX[11] == null || hl7BtsMessage.BTX[11] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						11, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					// ******************************************************************************
					// Possible Values:
					//			RA = Returned unused/no longer needed
					//			RL = Returned unused/keep linked to patient for possible use later
					//			TR = Transfused with adverse reaction
					//			TX = Transfused
					//			WA = Wasted (product no longer viable)
					// ******************************************************************************
					//	VBECS currently only supports:
					//			TX = Transfused
					//
					if ( HL7Utility.GetBloodProductTransfusionStatusEnumFromString(hl7BtsMessage.BTX[11]) != BloodProductTransfusionStatus.TX )
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
							11, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, string.Empty) );
					}
				}
				// *********************************************************************************************************************************************************************
				// BP Message Status
				if ( hl7BtsMessage.BTX.Length < 13 || hl7BtsMessage.BTX[12] == null || hl7BtsMessage.BTX[12] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						12, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					// ******************************************************************************
					// Possible Values:
					//			F = Final status; Can only be changed with a corrected status
					//			P = Preliminary status
					// ******************************************************************************
					//	VBECS currently only supports:
					//			F = Final status
					//	
					if ( HL7Utility.GetBloodProductMessageStatusEnumFromChar(hl7BtsMessage.BTX[12][0]) != BloodProductMessageStatus.Final )
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
							12, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, string.Empty) );
					}
				}
				// *********************************************************************************************************************************************************************
				// BP Date/Time of Status
				if ( hl7BtsMessage.BTX.Length < 14 || hl7BtsMessage.BTX[13] == null || hl7BtsMessage.BTX[13] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						13, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					try
					{
						if ( HL7DateFormat.ConvertHL7DateTime(hl7BtsMessage.BTX[13]) == DateTime.MinValue )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
								13, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch(Exception)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
							13, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
					}
				}
				// *********************************************************************************************************************************************************************
				// BP Administrator
				if ( hl7BtsMessage.BTX.Length < 15 || hl7BtsMessage.BTX[14] == null || hl7BtsMessage.BTX[14] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						14, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				// *********************************************************************************************************************************************************************
				// BP Transfusion Start Date/Time of Status
				if ( hl7BtsMessage.BTX.Length < 17 || hl7BtsMessage.BTX[16] == null || hl7BtsMessage.BTX[16] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						16, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					try
					{
						if ( HL7DateFormat.ConvertHL7DateTime(hl7BtsMessage.BTX[16]) == DateTime.MinValue )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
								16, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch(Exception)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
							16, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
					}
				}
				// *********************************************************************************************************************************************************************
				// BP Transfusion End Date/Time of Status
				if ( hl7BtsMessage.BTX.Length < 18 || hl7BtsMessage.BTX[17] == null || hl7BtsMessage.BTX[17] == string.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						17, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				else
				{
					try
					{
						if ( HL7DateFormat.ConvertHL7DateTime(hl7BtsMessage.BTX[17]) == DateTime.MinValue )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
								17, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
						}
					}
					catch(Exception)
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
							17, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
					}
				}
			}
			//
			return errSegment.ToString();
		}

		/// <summary>
		/// CR 3006: added this method to return a validated PatientTransfusionData object;
		/// if the return string is not empty then that means there were problems with the message data.
		/// </summary>
		/// <param name="hl7Interface"></param>
		/// <param name="hl7BtsMessage"></param>
		/// <param name="patientTransfusionData"></param>
		/// <returns></returns>
		private static string ValidatePatientTransfusionData(HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage, ref PatientTransfusionData patientTransfusionData)
		{
			string patientDfn = null;
			string patientSsn = null;
			string [] locationData = null;
			//
			// We can have 0 - 1 of each NTE type
			int typeReMax = 1;
			int typeVaBcrMax = 1;
			int typeReCount = 0;
			int typeVaBcrCount = 0;
			//
			StringBuilder errSegment = new StringBuilder();
			//
			// *********************************************************************************************************************************************************************
			// Patient IDs ******
			// ******************
			try
			{
				Convert.ToInt64(hl7BtsMessage.PID[3]);
                patientDfn = hl7BtsMessage.PID[3];
			}
			catch(Exception)
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
					3, 0, 0, AckErrorCodes.DataTypeError, AckErrorSeverity.Error, string.Empty) );
			}
			//
			try
			{
				patientSsn = hl7BtsMessage.PID[19];
				//
				System.Text.RegularExpressions.Regex ssn = Common.RegularExpressions.PatientSsn();
				//
				if ( !ssn.IsMatch(patientSsn) )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
						19, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, string.Empty) );
				}
			}
			catch (Exception ex)
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
					19, 0, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
			}
			//
			if ( patientDfn != null && patientSsn != null )
			{
				patientTransfusionData.PatientGuid = DAL.Patient.GetPatientGuidForPatient( Convert.ToInt64(patientDfn), patientSsn );
				//
				if ( patientTransfusionData.PatientGuid == Guid.Empty )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PID, SEQUENCE_PID, 
						0, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, "Patient not found in VBECS.") );
				}
			}
			// *********************************************************************************************************************************************************************
			// Patient Location Data ******
			// ****************************
			try
			{
				locationData = HL7Utility.ParseGetStringArray( hl7BtsMessage.PV1[3], hl7BtsMessage.Delimiters[1].ToString() );
				//
				// Assigned Patient Location: Location Name
				// PV1.3.1
				patientTransfusionData.LocationName = GetLocationDataFromArray( locationData, 0 );	
				//
				if ( patientTransfusionData.LocationName == null || patientTransfusionData.LocationName.Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PV1, SEQUENCE_PV1,
						3, 1, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
			}
			catch (Exception ex)
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PV1, SEQUENCE_PV1,
					3, 1, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
			}
			//
			// Assigned Patient Location: Division Code
			// PV1.3.10.1
			try
			{
				patientTransfusionData.DivisionCode = GetLocationDataFromArray( locationData, 9 );
				//
				if ( patientTransfusionData.DivisionCode == null || patientTransfusionData.DivisionCode.Length == 0 )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PV1, SEQUENCE_PV1,
						10, 1, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
			}
			catch (Exception ex)
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_PV1, SEQUENCE_PV1,
					10, 1, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
			}
			// *********************************************************************************************************************************************************************
			// Blood Unit Guid ******
			// **********************
			bool bloodUnitValid = true;
			try
			{
				patientTransfusionData.BloodUnitGuid = new Guid( hl7BtsMessage.BTX[2] );
				//
				// Make sure blood unit is not empty and is in VBECS
				bloodUnitValid = patientTransfusionData.BloodUnitGuid != Guid.Empty && BOL.BloodUnit.BloodUnitExists(patientTransfusionData.BloodUnitGuid);
			}
			catch(Exception)
			{
				bloodUnitValid = false;
			}
			if ( !bloodUnitValid )
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
					2, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, string.Empty) );
			}
			// *********************************************************************************************************************************************************************
			// Primary Transfusionist ******
			// *****************************
			try
			{
				string [] primaryTransfusionist = HL7Utility.ParseGetStringArray( hl7BtsMessage.BTX[14], hl7BtsMessage.Delimiters[1].ToString() );
				//
				string primaryTransfusionistLastName = HL7Utility.ConvertString( primaryTransfusionist[1] );
				string primaryTransfusionistFirstName = HL7Utility.ConvertString( primaryTransfusionist[2] );
				//
				if ( primaryTransfusionistLastName == null )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						14, 2, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				//
				if ( primaryTransfusionistFirstName == null )
				{
					errSegment.Append( BceBrtMessage.BuildErrSegment(
						hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
						14, 3, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
				}
				//
				patientTransfusionData.PrimaryTransfusionist = string.Concat( primaryTransfusionist[1], ", ", primaryTransfusionist[2] );
			}
			catch (Exception ex)
			{
				errSegment.Append( BceBrtMessage.BuildErrSegment(
					hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_BTX, SEQUENCE_BTX, 
					14, 3, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, HL7Utility.GetInnerExceptionFromException(ex)) );
			}
			// *********************************************************************************************************************************************************************
			// Comments ******
			// ***************
			if ( hl7BtsMessage.NTE != null && hl7BtsMessage.NTE.Count > 0 )
			{
				for ( int idx = 0; idx < hl7BtsMessage.NTE.Count; idx++ )
				{
					string [] nte = (string [])hl7BtsMessage.NTE[idx];
					//
					string commentTypeString = nte[4];
					//
					if ( commentTypeString == null || commentTypeString == string.Empty )
					{
						errSegment.Append( BceBrtMessage.BuildErrSegment(
							hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_NTE_RE, SEQUENCE_NTE_RE, 
							4, 0, 0, AckErrorCodes.RequiredFieldMissing, AckErrorSeverity.Error, string.Empty) );
					}
					else
					{
						// Get the comments, if any, for the appropriate types
						BloodProductCommentType commentType = HL7Utility.GetBloodProductCommentTypeEnumFromString( commentTypeString );
						//
						if ( commentType != BloodProductCommentType.Barcode && commentType != BloodProductCommentType.FreeText )
						{
							errSegment.Append( BceBrtMessage.BuildErrSegment(
								hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_NTE_RE, SEQUENCE_NTE_RE, 
								4, 0, 0, AckErrorCodes.UnknownKeyIdentifier, AckErrorSeverity.Error, "Invalid comment type.") );
						}
						else if ( commentType == BloodProductCommentType.Barcode )
						{
							if ( typeVaBcrCount <= typeVaBcrMax )
							{
								typeVaBcrCount += 1;
								patientTransfusionData.BedsideVerificationUnitManualEntryReason = nte[3];
							}
							else
							{
								errSegment.Append( BceBrtMessage.BuildErrSegment(
									hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_NTE_VA_BCR, SEQUENCE_NTE_VA_BCR, 
									4, 0, 0, AckErrorCodes.SegmentSequenceError, AckErrorSeverity.Error, "Unsupported number of segment types: only 1 type of 'VA-BCR' was expected.") );
							}
						}
						else if ( commentType == BloodProductCommentType.FreeText )
						{
							if ( typeReCount <= typeReMax )
							{
								typeReCount += 1;
								patientTransfusionData.PatientTransfusionComment = nte[3];
							}
							else
							{
								errSegment.Append( BceBrtMessage.BuildErrSegment(
									hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], ID_NTE_RE, SEQUENCE_NTE_RE, 
									4, 0, 0, AckErrorCodes.SegmentSequenceError, AckErrorSeverity.Error, "Unsupported number of segment types: only 1 type of 'RE' was expected.") );
							}
						}
					}
				}
			}
			//
			return errSegment.ToString();
		}

		#endregion

		#region Process Patient Transfusion Data

		/// <summary>
		/// Here we call LoadPatientTransfusionData() to fill BOL.PatientTransfusion object,
		/// attempt to lock record(s) and update VBECS with data.
		/// 
		/// Any exception(s) encountered will be handled in ParseHL7Message().
		/// </summary>
		private static string ProcessUpdatedData( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage, ref PatientTransfusionData patientTransfusionData )
		{
			StringBuilder errSegment = new StringBuilder();
			//
			patientTransfusionData.AckCode = AckCodes.AA;
			//
			bool lockOk = false;
			//
			Guid [] lockedRecords = null;
			//
			try
			{
				BOL.PatientTransfusion patientTransfusion = LoadPatientTransfusionData( hl7Interface, hl7BtsMessage, patientTransfusionData, ref errSegment );
				//
				if ( patientTransfusion != null )
				{
					Guid bloodUnitStatusGuid = (Guid)DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(patientTransfusionData.BloodUnitGuid).Rows[0][TABLE.BloodUnitStatus.BloodUnitStatusGuid];
					//
					if ( patientTransfusion.IsNew )
					{
						// Need to lock blood unit record if new
						lockedRecords = new Guid[]{ bloodUnitStatusGuid };
					}
					else
					{
						// Also need to lock patient transfusion record if editing
						lockedRecords = new Guid[]{ bloodUnitStatusGuid, patientTransfusion.PatientTransfusionGuid };
					}
					//
					lockOk = SetLocks( lockedRecords );
					//
					if ( lockOk )
					{
						patientTransfusion.Save( Common.WorkloadProcessID.TransfuseUnit );
					} 
					else
					{
						// Record is locked - send AE unless re-transmit attempts have been exhausted,
						// in this case we send the AR and email the interface adminstrator
						//
						patientTransfusionData.TransmitCount = HL7MessageLog.GetTransmitCountForMessage( hl7BtsMessage.MessageControlID );
						//
						if ( patientTransfusionData.TransmitCount <= hl7Interface.ReTransmitAttempts )
						{
							patientTransfusionData.AckCode = AckCodes.AE;
							patientTransfusionData.TransmitCount += 1;
							patientTransfusionData.ErrorMessage = RECORD_LOCKED;
							errSegment.Append( BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, 0, 0, AckErrorCodes.ApplicationRecordLocked, AckErrorSeverity.Error, RECORD_LOCKED) );
						}
						else
						{
							patientTransfusionData.AckCode = AckCodes.AR;
							patientTransfusionData.ErrorMessage = RECORD_LOCKED_MAX_ATTEMPTS_MADE;
							errSegment.Append( BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, 0, 0, AckErrorCodes.ApplicationRecordLocked, AckErrorSeverity.Error, RECORD_LOCKED_MAX_ATTEMPTS_MADE) );
						}
					}
				}
			}
			finally
			{
				if ( lockOk )
				{
					UnlockRecords( lockedRecords, LOCK_FORM_ID, Common.LockFunctions.UC069PostTransfusionInformation );
				}
			}
			//
			return errSegment.ToString();
		}

		/// <summary>
		/// Load up a BOL.PatientTransfusion object based on data from the HL7 message; 
		/// will return true if BrokenRulesCount == 0 
		/// 
		/// Any exception(s) encountered will be handled in ParseHL7Message()
		/// </summary>
		private static BOL.PatientTransfusion LoadPatientTransfusionData( HL7Interface hl7Interface, HL7BtsMessage hl7BtsMessage, PatientTransfusionData patientTransfusionData, ref StringBuilder errSegment )
		{
			Common.LogonUser.LogonUserDUZ = VBECS_USER;
			Common.LogonUser.LogonUserDivisionCode = patientTransfusionData.DivisionCode; 
			// 
			BOL.PatientTransfusion patientTransfusion = new BOL.PatientTransfusion( patientTransfusionData.PatientGuid, patientTransfusionData.BloodUnitGuid, false );
			//
			patientTransfusion.SetInitialBrokenRules( "FrmEnterPostTransfusionInfo", "default", true );
			//
			// We need to record this transfusion record was entered by the BCE COTS application
			patientTransfusion.ApplicationOfRecordId = Common.Utility.GetApplicationOfRecordTypeIntegerValue( Common.ApplicationOfRecordType.BCE_COTS );
			patientTransfusion.PatientLocation = patientTransfusionData.LocationName;
			patientTransfusion.TransfusedVolume = Convert.ToInt32( hl7BtsMessage.BTX[9] );
			//
            // Defect 210141 / CR 3211 
            patientTransfusion.BedsideVerificationDate = DAL.VbecsDateTime.ConvertToDivisionTime(hl7BtsMessage.BTX[13], patientTransfusionData.DivisionCode);
            patientTransfusion.TransfusionStartDateTime = DAL.VbecsDateTime.ConvertToDivisionTime(hl7BtsMessage.BTX[16], patientTransfusionData.DivisionCode);
            patientTransfusion.TransfusionEndDateTime = DAL.VbecsDateTime.ConvertToDivisionTime(hl7BtsMessage.BTX[17], patientTransfusionData.DivisionCode);
            //
			#region Transfusionists

			patientTransfusion.Transfusionist1Id = patientTransfusionData.PrimaryTransfusionist;
			//
			string [] secondaryTransfusionist = HL7Utility.ParseGetStringArray( hl7BtsMessage.BTX[15], hl7BtsMessage.Delimiters[1].ToString() );
			//
			// CR 3073
			if ( secondaryTransfusionist != null && secondaryTransfusionist.Length == 3 
				&& HL7Utility.ConvertString(secondaryTransfusionist[1]) != null && HL7Utility.ConvertString(secondaryTransfusionist[2]) != null )
			{
				patientTransfusion.Transfusionist2Id = secondaryTransfusionist[1] + ", " + secondaryTransfusionist[2];
			}

			#endregion
			//
			#region Transfusion Reaction 

			if ( hl7BtsMessage.BTX[18].Length > 0 )
			{
				string reactionIndicator = hl7BtsMessage.BTX[18].Substring(1);
				//
				patientTransfusion.ReactionIndicator = reactionIndicator == REACTION_AFFIRMATIVE_INDICATOR;
			}
			else
			{
				patientTransfusion.ReactionIndicator = false;
			}

			#endregion
			//
			#region Transfusion Reactions\Interruption Comments

			if ( hl7BtsMessage.BTX[19].Length > 0 )
			{
				patientTransfusion.TransfusionInterrupted = true; 
				//
				string fullInterruptionComments = hl7BtsMessage.BTX[19].Substring(1, hl7BtsMessage.BTX[19].Length - 1);
				//
				string [] interruptionComments = SplitInterruptionComments( fullInterruptionComments, patientTransfusionData.CommentSeparator, patientTransfusionData.ReactionMarker );
				//
				if ( interruptionComments != null && interruptionComments[0] != null && interruptionComments[0].Length > 0 )
				{
					string reactionSymptoms = interruptionComments[0];
					//
					if ( reactionSymptoms.Length > 510 )
					{
						reactionSymptoms = reactionSymptoms.Substring(0, 510);
					}
					//
					patientTransfusion.ReactionSymptoms = reactionSymptoms;
				}
				//
				if ( interruptionComments != null && interruptionComments[1] != null && interruptionComments[1].Length > 0 )
				{
					string interruptionComment = interruptionComments[1];
					//
					if ( interruptionComment.Length > 510 )
					{
						interruptionComment = interruptionComment.Substring(0, 510);
					}
					//
					patientTransfusion.InterruptionComment = interruptionComment;
				}
			}
			else
			{
				patientTransfusion.TransfusionInterrupted = false;
			}

			#endregion
			//
			#region NTE Segments

			// Save the comments, if any were found in ValidateBtxSegmentContent()
			//
			if ( patientTransfusionData.BedsideVerificationUnitManualEntryReason != null )
			{
				patientTransfusion.BedsideVerificationUnitManualEntryReason = patientTransfusionData.BedsideVerificationUnitManualEntryReason;
			}
			//
			if ( patientTransfusionData.PatientTransfusionComment != null )
			{
				patientTransfusion.PatientTransfusionComment = patientTransfusionData.PatientTransfusionComment;
			}

			#endregion
			//
			#region Handle Broken Rules (if any)

			// BR_69.23
			if ( patientTransfusion.BrokenRulesCount > 0 )
			{
				string errorMessage = patientTransfusion.GetBrokenRulesMessageSingleLine("FrmEnterPostTransfusionInfo");
				//
				patientTransfusionData.AckCode = AckCodes.AR;
				patientTransfusionData.ErrorMessage = errorMessage;
				//
				errSegment.Append( BceBrtMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, 0, 0, AckErrorCodes.ApplicationInternalError, AckErrorSeverity.Error, errorMessage) );
				//
				return null;
			}

			#endregion
			//
			return patientTransfusion;
		}

		#endregion

		#region Locking

		/// <summary>
		/// Create database record locks
		/// </summary>
		/// <param name="recordGuids"></param>
		/// <returns>Indicator if successful</returns>
		private static bool SetLocks(params System.Guid[] recordGuids)
		{
			foreach (System.Guid recordGuid in recordGuids)
			{
				if ( !LockRecord(recordGuid) )
				{
                    UnlockRecords(recordGuids, LOCK_FORM_ID, Common.LockFunctions.UC069PostTransfusionInformation);
					//
					return false;
				}
			}
			//
			return true;
		}

		/// <summary>
		/// Lock a Record
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <returns></returns>
		private static bool LockRecord(System.Guid recordGuid)
		{
			if (recordGuid == System.Guid.Empty || IsRecordAlreadyLockedInUC(recordGuid))
			{
				return false;
			}

            //CR3311 adding lockinactivitytimeout
            System.Data.DataTable dtLock = DAL.LockManager.LockRecord(recordGuid, LOCK_FORM_ID, Common.LockFunctions.UC069PostTransfusionInformation, false, LOCK_FORM_ID, BOL.LockManager.DefaultLockInactivityTimeoutMin);
			//
			if (dtLock.Rows.Count == 0 || dtLock.Rows.Count > 1)
			{
				return false;
			}
			//
			Lock tmpLock = new BOL.Lock(dtLock.Rows[0]);
			//
			// Check if there was a conflict
			if (tmpLock.LockConflictIndicator(LOCK_FORM_ID))
			{
				return CheckLockConflicts(tmpLock);
			}
			//
			return true;
		}

		/// <summary>
		/// TODO: TEST THIS.. I'm not sure if David is doing it right - LOHSE
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <returns></returns>
		private static bool IsRecordAlreadyLockedInUC(System.Guid recordGuid)
		{
		    BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid, false);

            if (tmpLock.LockExists)
            {
                //TODO: maybe make a IsLockMine Property on Lock class?
                if (tmpLock.LockedRecordGuid == recordGuid && tmpLock.LockedFormId == LOCK_FORM_ID && tmpLock.SessionGuid == Common.LogonUser.VbecsSessionGuid)
                    return (true);
            }

            return (false);
		}

		/// <summary>
		/// UnlockRecords
		/// </summary>
		private static void UnlockRecords(Guid [] recordGuids, int lockedFormId, Common.LockFunctions lockFunction )
		{
			DAL.LockManager.UnlockRecords(recordGuids, LOCK_FORM_ID ,lockFunction );
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="tmpLock"></param>
		private static bool CheckLockConflicts(Lock tmpLock)
		{
			// Have to do ToUpper Cuz DB and Windows might not be EXACTLY da same
			if (tmpLock.UserNTLoginId.ToUpper() != Common.LogonUser.LogonUserName.ToUpper())
			{
				return false;
			}

            if (tmpLock.LockFunctionId == Common.LockFunctions.UC069PostTransfusionInformation && tmpLock.LockedFormId == LOCK_FORM_ID)  
			{
				// For all NON UC_38 locks, if the session is different, it's a same record lock
				if (tmpLock.SessionGuid != Common.LogonUser.VbecsSessionGuid)
				{
					return false;
				}
			}
			else
			{
				return false;
			}
			//
			return true;
		}

		#endregion 

		#region Helper Methods

		/// <summary>
		/// Splits a string (properly) into n substrings based on delimiters.
		/// Returns two strings: [0] reactions, [1] other interruption reasons
		/// </summary>
		private static string [] SplitInterruptionComments( string toSplit, string commentSeparator, char reactionMarker )
		{
			StringBuilder reactions = new StringBuilder();
			StringBuilder interruptionReasons = new StringBuilder();

			System.Text.RegularExpressions.MatchCollection matches = System.Text.RegularExpressions.Regex.Matches(toSplit, REGEXPATTERN);
			foreach( System.Text.RegularExpressions.Match m in matches )
			{
				// Are empty matches possible?
				if(m.Value == null || m.Value.Length == 0)
				{
					continue;
				}
				// Select appropriate builder
				StringBuilder strBuilder = (m.Value[0] == reactionMarker ? reactions : interruptionReasons);
				// Add parsed line (remove reaction marker if necessary)
				strBuilder.Append(m.Value[0] == reactionMarker ? m.Value.Substring(1, m.Value.Length - 1) : m.Value ).Append(commentSeparator);
			}
			if (reactions.Length > 0)
			{
				reactions.Length -= 1;
			}
			if (interruptionReasons.Length > 0)
			{
				interruptionReasons.Length -= 1;
			}
			return new string []{ reactions.ToString(), interruptionReasons.ToString() };
		}

		/// <summary>
		///  
		/// </summary>
		private static string GetLocationDataFromArray( string [] locationData, int index)
		{
			if ( locationData != null 
				&& locationData.Length >= (index + 1) 
				&& locationData[index] != null 
				&& locationData[index].Length > 0 
				&& locationData[index] != EMPTY_LOCATION_1 
				&& locationData[index] != EMPTY_LOCATION_2 )
			{
				return locationData[index];
			}
			else
			{
				return null;
			}
		}

		#endregion
		
		#endregion
	}
}
